/* ***************************************************************** 
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2004 Sandia Corporation and Argonne National
    Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
    with Sandia Corporation, the U.S. Government retains certain 
    rights in this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License 
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
    pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov      
   
  ***************************************************************** */
// -*- Mode : c++; tab-width: 3; c-tab-always-indent: t; indent-tabs-mode: nil; c-basic-offset: 3 -*-

/*! \file CompositeOFMultiply.hpp

Header file for the Mesquite:: CompositeOFMultiply class

  \author Michael Brewer
  \date   2002-05-23
 */


#ifndef CompositeOFExponent_hpp
#define CompositeOFExponent_hpp

#include "Mesquite.hpp"
#include "ObjectiveFunction.hpp"
#include "MsqHessian.hpp"
#include "MsqTimer.hpp"
#include "PatchData.hpp"

namespace MESQUITE_NS
{
   /*!\class CompositeOFMultiply
       \brief Multiplies two ObjectiveFunction values together.
     */
   class MsqMeshEntity;
   class PatchData;
   class CompositeOFExponent : public ObjectiveFunction
   {
   public:
	 MESQUITE_EXPORT
     CompositeOFExponent(ObjectiveFunction* Obj,
					  bool delete_OFs = false)
	   : deleteObjFuncs(delete_OFs)
	{
		objFunc = Obj;
	}
	 MESQUITE_EXPORT
     virtual ~CompositeOFExponent(){
	  if (deleteObjFuncs) {
		delete objFunc;
	  }
	}
     
      //!\brief Called at start of instruction queue processing
      //!
      //! Do any preliminary global initialization, consistency checking,
      //! etc. 
/*      virtual void initialize_queue( Mesh* mesh,
                                    MeshDomain* domain,
                                    const Settings* settings,
                                    MsqError& err )
	{
	  objFunc->initialize_queue( mesh, domain, settings, err ); MSQ_ERRRTN(err);
	} */

     virtual bool initialize_block_coordinate_descent( Mesh* mesh, 
                                                       MeshDomain* domain,
                                                       const Settings* settings,
                                                       PatchSet* user_set,
                                                       MsqError& err )
	{
	  bool rval;
	  rval = objFunc->initialize_block_coordinate_descent( mesh, domain, settings, user_set, err );
	  MSQ_ERRZERO(err);
	  return !MSQ_CHKERR(err) && rval;
	}											

     virtual bool evaluate( EvalType type, 
                            PatchData& pd,
                            double& value_out,
                            bool free,
                            MsqError& err )
	{
	  bool ok;
	  
	  ok = objFunc->evaluate( type, pd, value_out, free, err );
	  if (MSQ_CHKERR(err) || !ok) return false;
	  value_out = exp(value_out);
	  return true;
	}							

     virtual bool evaluate_with_gradient( EvalType type, 
                                          PatchData& pd,
                                          double& value_out,
                                          std::vector<Vector3D>& grad_out,
                                          MsqError& err )
	{
	  bool ok;
	  ok = objFunc->evaluate_with_gradient( type, pd, value_out, grad_out, err );
	  if (MSQ_CHKERR(err) || !ok) return false;
	  assert( grad_out.size() == pd.num_free_vertices() );
	  value_out = exp(value_out);
	  for (std::vector<Vector3D>::iterator i = grad_out.begin(); i != grad_out.end(); ++i)
		*i *= value_out;
	  return true;
	}										  

     /* virtual bool evaluate_with_Hessian_diagonal( EvalType type, 
                                        PatchData& pd,
                                        double& value_out,
                                        std::vector<Vector3D>& grad_out,
                                        std::vector<SymMatrix3D>& hess_diag_out,
                                        MsqError& err ); 
    
     virtual bool evaluate_with_Hessian( EvalType type, 
                                         PatchData& pd,
                                         double& value_out,
                                         std::vector<Vector3D>& grad_out,
                                         MsqHessian& Hessian_out,
                                         MsqError& err );  */

     virtual ObjectiveFunction* clone() const
	 { return new CompositeOFExponent( objFunc->clone(), true ); }
	 
     virtual void clear()
	 {
		objFunc->clear();
     }
     
     virtual int min_patch_layers() const
	 {
		return objFunc->min_patch_layers();
	 }
     
	private:
          
     ObjectiveFunction* objFunc;
	 Mesquite::Exponent mPower;
     bool deleteObjFuncs;
   };
}//namespace
#endif //  CompositeOFMultiply_hpp



